<?php

namespace App\Http\Controllers;

use App\ImageUpload;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
class ServiceController extends Controller
{
    use ImageUpload;
    /**
     * Display a listing of the blogs.
     */
    public function index()
    {
        $services = Service::paginate(10); // Retrieve all blogs
        return view('service.index', compact('services')); // Return the view with blogs
    }

    /**
     * Show the form for creating a new blog.
     */
    public function create()
    {
        return view('service.create'); // Return the create form view
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg',
        ]);

        if ($validator->fails()) {
            // Flash each error message to the session
            foreach ($validator->errors()->all() as $error) {

                flash()->error($error);
            }

            // Redirect back with input and errors
            return redirect()->back()->withInput()->withErrors($validator);
        }

        // Proceed with storing the blog post...
        $service = new Service();
        $service->title = $request->input('title');
        $service->content = $request->input('content');

        if ($request->hasFile('photo')) {
            $image = $request->file('photo');
            $imageName = time() . '.' .  $image->getClientOriginalExtension();
            $image->move(public_path('uploads'), $imageName);
            $service->photo = 'uploads/' . $imageName;
        }

        $service->save();
        flash()->success('Article created successfully.');

        return redirect()->back();
    }

    /**
     * Display the specified blog.
     */
    public function show(Service $service)
    {
        return view('service', compact('service')); // Show the specific blog
    }

    /**
     * Show the form for editing the specified blog.
     */
    public function edit(Service $service)
    {
        return view('service.edit', compact('service')); // Return the edit form view
    }

    /**
     * Update the specified blog in the database.
     */
    public function update(Request $request, Service $service)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg',
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->all() as $error) {
                flash()->error($error);
            }

            return redirect()->back()->withInput()->withErrors($validator);
        }

        $service->title = $request->input('title');
        $service->content = $request->input('content');

        if ($request->hasFile('photo')) {
            // Optionally delete the old photo if it exists
            if ($service->photo && file_exists(public_path($service->photo))) {
                unlink(public_path($service->photo));
            }

            $image = $request->file('photo');
            $imageName = time() . '.' .  $image->getClientOriginalExtension();
            $image->move(public_path('uploads'), $imageName);
            $service->photo = 'uploads/' . $imageName;
        }

        $service->save();

        flash()->success('Article updated successfully.');
        return redirect()->back();
    }


    /**
     * Remove the specified blog from the database.
     */
    public function destroy(Service $service)
    {

        $service->delete(); // Auto deletes the image when the blog is deleted
        flash()->success('Article deleted successfully.');
        return back(); // Redirect to the blog index
    }
}
